'use strict'

/*
todo:
more undo actions
getUndoDescription?
undo groups
second constructor that always/only takes data
*/

const commandStack = []
const undoStack = []
let redoStack = []

function addCommand(command) {
    commandStack.push(command)
}

let lastProcessedCommandTime = 0

function updateCommands () {
    while (commandStack.length) {
        redoStack = []

        const command = commandStack.shift()

        const c1 = JSON.stringify(command.serialize())
        const c2 = deserializeCommand(JSON.parse(c1))
        const undoCommand = c2.makeUndo(entities)
        c2.do(entities, false)

        devserverSend({
            type: 'command',
            data: command.serialize()
        })

        if (undoCommand) {
            const timeSinceLastCommandProcessed = Date.now() - lastProcessedCommandTime
            if (timeSinceLastCommandProcessed < 1000 && undoStack.length) {
                const foldedCommand = command.tryFold(undoStack[0])
                if (foldedCommand) {
                    undoStack[0] = foldedCommand
                } else {
                    undoStack.unshift(undoCommand)
                }
            } else {
                undoStack.unshift(undoCommand)
            }
        }
        lastProcessedCommandTime = Date.now()
    }
}

function undoCommand () {
    if (undoStack.length) {
        const command = undoStack.shift()

        const c1 = JSON.stringify(command.serialize())
        const c2 = deserializeCommand(JSON.parse(c1))
        const redoCommand = c2.makeUndo(entities)
        c2.do(entities, false)

        devserverSend({
            type: 'command',
            data: command.serialize()
        })

        if (redoCommand) {
            redoStack.push(redoCommand)
        }
    }
}

function redoCommand () {
    if (redoStack.length) {
        const command = redoStack.pop()

        const c1 = JSON.stringify(command.serialize())
        const c2 = deserializeCommand(JSON.parse(c1))
        const undoCommand = c2.makeUndo(entities)
        c2.do(entities, false)

        devserverSend({
            type: 'command',
            data: command.serialize()
        })

        if (undoCommand) {
            undoStack.unshift(undoCommand)
        }
    }
}

function deserializeCommand (data) {
    const type = data.type
    data = data.data
    let command = null
    switch (type) {
        case 'set-name': command = new CommandSetName(); break
        case 'create-entities': command = new CommandCreateEntities(); break
        case 'delete-entities': command = new CommandDeleteEntities(); break
        case 'set-static-value': command = new CommandSetStaticValue(); break
        case 'set-dynamic-type': command = new CommandSetDynamicType(); break
        case 'set-dynamic-value': command = new CommandSetDynamicValue(); break
        case 'add-dynamic-key': command = new CommandAddDynamicKey(); break
        case 'remove-dynamic-key': command = new CommandRemoveDynamicKey(); break
        case 'set-dynamic-key': command = new CommandSetDynamicKey(); break
        case 'set-dynamic-expression': command = new CommandSetDynamicExpression(); break
        case 'move-modules-parents': command = new CommandMoveModulesParents(); break
        case 'move-module-order': command = new CommandMoveModuleOrder(); break
    }
    command.data = data
    return command
}

class Command {
    constructor (commandType) {
        if (!commandStack) throw new Error('no name')
        this.commandType = commandType
    }
    do (entities, isServer) {
        console.error('do() not implemented in command')
    }
    tryFold(command) {
        return undefined
    }
    makeUndo (entities) {
        return undefined
    }
    serialize () {
        const data = {
            type: this.commandType,
            data: this.data
        }
        return data
    }
}

/*
{
    uuid?
    type
    subType
    name
    config or {}
    parentId?
    insertPosition?
}
*/
class CommandCreateEntities extends Command {
    constructor (createConfigs=[]) { // ugh.. deserializeCommand calls this with undefined
        super('create-entities')
        this.data = {
            createConfigs: createConfigs.map((createConfig) => {
                let uuid
                try {
                    uuid = crypto.randomUUID()
                } catch (e) {
                }
                return {
                    uuid: createConfig.uuid || uuid,
                    type: createConfig.type,
                    subType: createConfig.subType,
                    name: createConfig.name,
                    config: createConfig.config,
                    parentId: createConfig.parentId,
                    insertPosition: createConfig.insertPosition,
                }
            })
        }
    }

    makeUndo (entities) {
        return new CommandDeleteEntities(this.data.createConfigs.map((createConfig) => createConfig.uuid))
    }

    do (entities, isServer) {
        this.data.createConfigs.forEach((createConfig) => {
            console.assert(entities[createConfig.uuid] === undefined)

            if (createConfig.parentId) {
                const toEntity = entities[createConfig.parentId]
                toEntity.config.modules = toEntity.config.modules || []
                if (createConfig.insertPosition === undefined) {
                    toEntity.config.modules.push(createConfig.uuid)
                } else {
                    toEntity.config.modules.splice(createConfig.insertPosition, 0, createConfig.uuid)
                }
            }

            entities[createConfig.uuid] = {
                name: createConfig.name,
                type: createConfig.type,
                subType: createConfig.subType,
                config: createConfig.config || {}
            }
            if (!isServer) {
                EntityAccessHelper.fromUuid(createConfig.uuid).init()
            }
        })
    }
}

class CommandSetName extends Command {
    constructor (entityId, name) {
        super('set-name')
        this.data = {
            entityId,
            name,
        }
    }

    tryFold(command) {
        if (command.commandType === this.commandType &&
            command.data.entityId === this.data.entityId) {
            this.data.name = command.data.name
            return this
        } else {
            return undefined
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const currentName = entity.name
        return new CommandSetName(this.data.entityId, currentName)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        entity.name = this.data.name
    }
}

/*
parentId?
depth
position
*/
function getEntityHierarchyDefs (entities) {
    // Find direct parent for each entity
    const parentIds = {}
    function recurse(entityId, parentId) {
        if (!parentIds[entityId]) {
            parentIds[entityId] = parentId
            const entity = entities[entityId]
            if (entity.config?.modules) {
                entity.config.modules.forEach(childId => {
                    recurse(childId, entityId)
                })
            }
        }
    }
    Object.keys(entities).forEach((entityId) => {
        recurse(entityId, undefined)
    })

    const entityDefs = {}
    function addEntityDef(entityId, parentId, depth, position) {
        if (!entityDefs[entityId] || entityDefs[entityId.depth < depth]) {
            entityDefs[entityId] = {parentId, depth, position}
        }
    }
    function dive(entityId, parentId, depth, childPos) {
        const entity = entities[entityId]
        addEntityDef(entityId, parentId, depth, childPos)
        if (entity.config?.modules) {
            let childPos = 0
            entity.config.modules.forEach(childId => {
                dive(childId, entityId, depth+1, childPos++)
            })
        }
    }
    let childPos = 0
    Object.keys(entities).forEach((entityId) => {
        if (parentIds[entityId] === undefined) {
            dive(entityId, undefined, 0, childPos++)
        }
    })

    return entityDefs
}

function gatherEntitiesForDeletion(entities, entityIds, deepestFirst) {
    // Recursively gather entities to be deleted
    const entitiesToDelete = new Set()
    function recurse(entityIds) {
        entityIds.forEach((entityId) => {
            entitiesToDelete.add(entityId)
            const entity = entities[entityId]
            if (entity.config.modules) {
                recurse(entity.config.modules)    
            }
        })
    }
    recurse(entityIds)

    const entityHierarchyDefs = getEntityHierarchyDefs(entities)
    return sortEntities(entityHierarchyDefs, entitiesToDelete, deepestFirst)
}

/*
entityDef
    entityId
    parentEntityId
    childPosition
*/
function sortEntities (entityHierarchyDefs, entityList, deepestFirst) {
    // Sort into correct order for deletion
    const sortedEntityList = [...entityList]
    sortedEntityList.sort((a, b) => {
        const aH = entityHierarchyDefs[a]
        const bH = entityHierarchyDefs[b]
        if (aH.depth !== bH.depth) return bH.depth - aH.depth // Deepest first, then..
        else return bH.position - aH.position // ..highest position first
    })
    if (!deepestFirst) {
        sortedEntityList.reverse()
    }
    return sortedEntityList.map((entityId) => {
        const entityHierarchyDef = entityHierarchyDefs[entityId]
        return {
            entityId,
            parentEntityId: entityHierarchyDef.parentId,
            childPosition: entityHierarchyDef.position,
        }
    })
}

class CommandDeleteEntities extends Command {
    constructor (entityIds) {
        super('delete-entities')
        this.data = {
            entityIds
        }
    }

    makeUndo (entities) {
        const creations = gatherEntitiesForDeletion(entities, this.data.entityIds, false)
        const createConfigs = []
        creations.forEach(deletion => {
            const { entityId, parentEntityId, childPosition } = deletion
            const entity = entities[entityId]
            const newCreateConfig = {
                uuid: entityId,
                type: entity.type,
                subType: entity.subType,
                name: entity.name,
                config: entity.config,
                parentId: parentEntityId,
                insertPosition: childPosition,
            }
            createConfigs.push(newCreateConfig)
        })

        return new CommandCreateEntities(createConfigs)
    }

    do (entities, isServer) {
        const deletions = gatherEntitiesForDeletion(entities, this.data.entityIds, true)
        deletions.forEach(deletion => {
            const { parentEntityId, entityId } = deletion
            if (parentEntityId) {
                entities[parentEntityId].config.modules = entities[parentEntityId].config.modules.filter(x => x !== entityId)
            }
            delete entities[entityId]
        })
    }
}

class CommandSetStaticValue extends Command {
    constructor (entityId, paramName, value) {
        super('set-static-value')
        this.data = {
            entityId,
            paramName,
            value,
        }
    }

    tryFold (command) {
        if (command.commandType === this.commandType &&
            command.data.entityId === this.data.entityId &&
            command.data.paramName === this.data.paramName) {
            this.data.value = command.data.value
            return this
        } else {
            return undefined
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const currentValue = entity.config[this.data.paramName]
        return new CommandSetStaticValue(this.data.entityId, this.data.paramName, currentValue)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        entity.config[this.data.paramName] = this.data.value

        if (!isServer) {
            const entityType = entity.type
            const entityTypeDef = entityRegistry[entityType]['_base']
            const typeConfigEntry = entityTypeDef.staticConfig.find(x => x.paramName === this.data.paramName)

            const entitySubType = entity.subType
            const entitySubTypeDef = entityRegistry[entityType][entitySubType]
            const subTypeConfigEntry = entitySubTypeDef.staticConfig.find(x => x.paramName === this.data.paramName)

            if (typeConfigEntry?.triggerInit || subTypeConfigEntry?.triggerInit) {
                EntityAccessHelper.fromEntity(entity).init()
            }
        }
    }
}

class CommandSetDynamicType extends Command {
    constructor (entityId, paramName, type, defaultValue) {
        super('set-dynamic-type')
        this.data = {
            entityId,
            paramName,
            type,
            defaultValue,
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const currentType = entity.config[this.data.paramName]?.type || 'value'
        return new CommandSetDynamicType(this.data.entityId, this.data.paramName, currentType, this.data.defaultValue)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        if (!entity.config[this.data.paramName]) {
            entity.config[this.data.paramName] = {}
        }
        if (entity.config[this.data.paramName]?.type !== this.data.type) {
            entity.config[this.data.paramName] = entity.config[this.data.paramName] || {}
            entity.config[this.data.paramName].type = this.data.type
            if (this.data.type === 'value' && entity.config[this.data.paramName].value === undefined) {
                entity.config[this.data.paramName].value = this.data.defaultValue
            } else if (this.data.type === 'keys' && entity.config[this.data.paramName].keys === undefined) {
                entity.config[this.data.paramName].keys = [[0, this.data.defaultValue]]
            }
        }
    }
}

class CommandSetDynamicValue extends Command {
    constructor (entityId, paramName, value) {
        super('set-dynamic-value')
        this.data = {
            entityId,
            paramName,
            value
        }
    }

    tryFold (command) {
        if (command.commandType === this.commandType &&
            command.data.entityId === this.data.entityId &&
            command.data.paramName === this.data.paramName) {
            this.data.value = command.data.value
            return this
        } else {
            return undefined
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const currentValue = entity.config[this.data.paramName].value
        return new CommandSetDynamicValue(this.data.entityId, this.data.paramName, currentValue)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        if (!entity.config[this.data.paramName]) {
            entity.config[this.data.paramName] = {}
        }
        entity.config[this.data.paramName].type = 'value'
        entity.config[this.data.paramName].value = this.data.value
    }
}

class CommandAddDynamicKey extends Command {
    constructor (entityId, paramName, insertAfter, time=undefined, value=undefined, interpolation=undefined) {
        super('add-dynamic-key')
        this.data = {
            entityId,
            paramName,
            insertAfter,
            time,
            value,
            interpolation,
        }
    }

    makeUndo (entities) {
        return new CommandRemoveDynamicKey(this.data.entityId, this.data.paramName, this.data.insertAfter + 1)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        let newKey
        if (this.data.insertAfter >= 0) {
            newKey = JSON.parse(JSON.stringify(entity.config[this.data.paramName].keys[this.data.insertAfter]))
        } else {
            newKey = JSON.parse(JSON.stringify(entity.config[this.data.paramName].keys[0]))
        }
        if (this.data.time !== undefined) {
            newKey[0] = this.data.time
        }
        if (this.data.value !== undefined) {
            newKey[1] = this.data.value
        }
        if (this.data.interpolation !== undefined) {
            newKey[2] = this.data.interpolation
        }
        entity.config[this.data.paramName].keys.splice(this.data.insertAfter+1, 0, newKey)
    }
}

class CommandRemoveDynamicKey extends Command {
    constructor (entityId, paramName, key) {
        super('remove-dynamic-key')
        this.data = {
            entityId,
            paramName,
            key,
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const key = entity.config[this.data.paramName].keys[this.data.key]
        return new CommandAddDynamicKey(this.data.entityId, this.data.paramName, this.data.key-1, key[0], key[1], key[2])
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        entity.config[this.data.paramName].keys.splice(this.data.key, 1)
    }
}

class CommandSetDynamicKey extends Command {
    constructor (entityId, paramName, key, time=undefined, value=undefined, interpolation=undefined) {
        super('set-dynamic-key')
        this.data = {
            entityId,
            paramName,
            key,
            time,
            value,
            interpolation,
        }
    }

    tryFold (command) {
        if (command.commandType === this.commandType &&
            command.data.entityId === this.data.entityId &&
            command.data.paramName === this.data.paramName &&
            command.data.key === this.data.key) {
            this.data.time = command.data.time !== undefined ? command.data.time : this.data.time
            this.data.value = command.data.value !== undefined ? command.data.value : this.data.value
            this.data.interpolation = command.data.interpolation !== undefined ? command.data.interpolation : this.data.interpolation
            return this
        } else {
            return undefined
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const key = entity.config[this.data.paramName].keys[this.data.key]
        return new CommandSetDynamicKey(this.data.entityId, this.data.paramName, this.data.key, key[0], key[1], key[2])
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        const key = entity.config[this.data.paramName].keys[this.data.key]
        if (this.data.time !== undefined) {
            key[0] = this.data.time
        }
        if (this.data.value !== undefined) {
            key[1] = this.data.value
        }
        if (this.data.interpolation !== undefined) {
            key[2] = this.data.interpolation
        }
    }
}


class CommandSetDynamicExpression extends Command {
    constructor (entityId, paramName, expression) {
        super('set-dynamic-expression')
        this.data = {
            entityId,
            paramName,
            expression,
        }
    }

    makeUndo (entities) {
        const entity = entities[this.data.entityId]
        const currentExpression = entity.config[this.data.paramName]?.expression
        return new CommandSetDynamicExpression(this.data.entityId, this.data.paramName, currentExpression)
    }

    do (entities, isServer) {
        const entity = entities[this.data.entityId]
        if (!entity.config[this.data.paramName]) {
            entity.config[this.data.paramName] = {}
        }
        if (this.data.expression) {
            entity.config[this.data.paramName].expression = this.data.expression
        } else {
            delete entity.config[this.data.paramName].expression
        }
    }
}


/**
 * entityId
 * fromId
 * toId
 * insertPosition?
 */
class CommandMoveModulesParents extends Command {
    constructor (moves) {
        super('move-modules-parents')
        this.data = {
            moves,
        }
    }

    makeUndo (entities) {
        const entityList = this.data.moves.map((x) => x.entityId)
        const entityHierarchyDefs = getEntityHierarchyDefs(entities)
        const sortedEntityList = sortEntities(entityHierarchyDefs, entityList, false)

        const undoMoves = []
        sortedEntityList.forEach((sortedEntityDef) => {
            const entityId = sortedEntityDef.entityId
            const move = this.data.moves.find(x => entityId === x.entityId)
            undoMoves.push({
                entityId: move.entityId,
                fromId: move.toId,
                toId: move.fromId,
                insertPosition: entityHierarchyDefs[move.entityId].position
            })
        })
        return new CommandMoveModulesParents(undoMoves)
    }

    do (entities, isServer) {
        // First remove from old parents
        this.data.moves.forEach(move => {
            const {entityId, fromId, toId} = {...move}
            const entity = entities[entityId]

            const fromEntity = entities[fromId]
            let fromStart = 0
            if (fromEntity) {
                fromStart = fromEntity.config.start || 0
                fromEntity.config.modules = (fromEntity.config.modules || []).filter(x => x !== entityId)
            }

            // Fixup start time
            const toEntity = entities[toId]
            const toStart = toEntity.config.start || 0
            entity.config.start = (entity.config.start || 0) + fromStart - toStart
        })

        // Now add to new parents
        this.data.moves.forEach(move => {
            const {entityId, toId, insertPosition} = {...move}

            const toEntity = entities[toId]
            toEntity.config.modules = toEntity.config.modules || []
            if (insertPosition !== undefined) {
                toEntity.config.modules.splice(insertPosition, 0, entityId)
            } else {
                toEntity.config.modules.push(entityId)
            }
        })
    }
}

class CommandMoveModuleOrder extends Command {
    constructor (entityId, move, toIndex) {
        super('move-module-order')
        this.data = {
            entityId,
            move,
            toIndex,
        }
    }

    makeUndo (entities) {
        // Find parent
        const entityHierarchyDefs = getEntityHierarchyDefs(entities)
        const parentEntityId = entityHierarchyDefs[this.data.entityId].parentId
        const parentEntity = entities[parentEntityId]
        const modules = parentEntity.config.modules
        const currentIndex = modules.indexOf(this.data.entityId)
        return new CommandMoveModuleOrder(this.data.entityId, 'toIndex', currentIndex)        
    }

    do (entities, isServer) {
        // Find parent
        const entityHierarchyDefs = getEntityHierarchyDefs(entities)
        const parentEntityId = entityHierarchyDefs[this.data.entityId].parentId
        const parentEntity = entities[parentEntityId]

        // Move
        const modules = parentEntity.config.modules
        const currentIndex = modules.indexOf(this.data.entityId)
        const currentItem = modules.splice(currentIndex, 1)[0]
        switch (this.data.move) {
            case 'top':
                modules.unshift(currentItem)
                break
            case 'bottom':
                modules.push(currentItem)
                break
            case 'up':
                modules.splice(Math.max(currentIndex - 1, 0), 0, currentItem)
                break
            case 'down':
                modules.splice(currentIndex + 1, 0, currentItem)
                break
            case 'toIndex':
                modules.splice(this.data.toIndex, 0, currentItem)
                break
        }
    }
}

try {
    module.exports = {
        CommandMoveModuleOrder,
        CommandMoveModulesParents,
        CommandSetName,
        deserializeCommand,
    }
} catch (e) {
}
